<?php

namespace App\Http\Controllers;

use App\Models\Country;
use App\Http\Requests\StoreCountryRequest;
use App\Http\Requests\UpdateCountryRequest;

use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Arr;
use Illuminate\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Route;

class CountryController extends Controller
{


    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('permission:view-country|create-country|edit-country|delete-country', ['only' => ['index', 'show']]);
        $this->middleware('permission:create-country', ['only' => ['create', 'store']]);
        $this->middleware('permission:edit-country', ['only' => ['edit', 'update']]);
        $this->middleware('permission:delete-country', ['only' => ['destroy']]);
    }


    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        user_logs();
        $data = Country::latest()->paginate(itemsPerPage());
        return view('countries.index', compact('data'))
            ->with('i', ($request->input('page', 1) - 1) * 5);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        user_logs();
        return view('countries.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreCountryRequest $request)
    {
        user_logs();
        $input = $request->all();
        Country::create($input);
        return redirect()->route('country.index')->with([
            'message' => 'Country created successfully',
            'alert-type' => 'success'
        ]);
    }


    public function statusUpdate(Request $request)
    {

        $response = null;
        $input = $request->all();
        $countryId = $input['countryId'];
        $country = Country::find($countryId);

        if ($country->status) {
            $country->status = Country::DISABLED;
            $response = 'disabledCountry';
        } else {
            $country->status = Country::ACTIVE;
            $response = 'activeCountry';
        }
        $country->save();
        return $response;
    }

    /**
     * Display the specified resource.
     */
    public function show(Country $country, $id)
    {
        user_logs();
        $countries = Country::find($id);
        return view('countries.show', compact('countries'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Country $country)
    {
        user_logs();
        $country = Country::find($country->id);
        return view('countries.edit', compact('country'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateCountryRequest $request, Country $country)
    {
        user_logs();
        $input = $request->all();
        $country = Country::find($country->id);
        $country->update($input);
        return redirect()->route('country.index')->with([
            'message' => 'Country updated successfully',
            'alert-type' => 'success'
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Country $country)
    {
        user_logs();
        Country::find($country->id)->delete();
        return redirect()->route('country.index')->with([
            'message' => 'Country deleted successfully',
            'alert-type' => 'success'
        ]);
    }
}
